﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.Manager;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    class FiveStarShooterEntity : EnemyEntity
    {
        enum eState
        {
            eStateGround,
            eStateShoot,
            eStateNone,
            eStateCount
        }

        /// <summary>
        /// Shooter's inner state
        /// </summary>
        eState _innerState;

        /// <summary>
        /// Texture when the shooter is on the ground
        /// </summary>
        int _iIdTextureGround;

        /// <summary>
        /// Sprite when the shooter is on the ground
        /// </summary>
        int _iIdSpriteGround;

        /// <summary>
        /// Shooter's animations
        /// </summary>
        int _iIdAnimation;

        /// <summary>
        /// Bullet's texture
        /// </summary>
        int _iIdTextureBullet;

        /// <summary>
        /// Bullet's sprite
        /// </summary>
        int _iIdSpriteBullet;

        /// <summary>
        /// Shape used when the shooter is on the ground
        /// </summary>
        StaticShapeRectangle _shapeGround;

        /// <summary>
        /// Shape used when the shooter is shooting
        /// </summary>
        StaticShapeRectangle _shapeShoot;

        /// <summary>
        /// Position of the shooters when in shooting state
        /// </summary>
        Vector2 _v2ShootingPosition;

        /// <summary>
        /// Bullet's spawn position
        /// </summary>
        Vector2 _v2BulletSpawnPosition;

        bool _bPreState;

        int _iGroundTime;
        int _iGroundStartTime;
        bool _bAnimationOver;
        int _iAnimationCount;
        bool _bShoot;
        int _iAnimationCurrentFrame;
        int _iAnimationCurrentTime;

        /// <summary>
        /// Id of the explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        /// <summary>
        /// Position offset for the explosion animation
        /// </summary>
        Vector2 _v2AnimationExplosionOffset;

        #region Properties
#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml"; } }
        public static string EDITOR_SPRITE { get { return "five_star_shooters_ground"; } }
#endif
        public int GroundTime { set { _iGroundTime = value; } }

        

        #endregion

        public override void deactivate()
        {
            base.Active = false;
            _shapeGround._bCollisionEnable = false;
            _shapeShoot._bCollisionEnable = false;

        }

        /// <summary>
        /// Constructor
        /// </summary>
        public FiveStarShooterEntity()
            :base()
        {
            _innerState = eState.eStateNone;
            _iIdTextureGround = -1;
            _iIdSpriteGround = -1;
            _iIdAnimation = -1;
            _iAnimationCurrentFrame = -1;
            _iAnimationCurrentTime = -1;
            _iIdAnimationExplosion = -1;
            _side = eSide.eSideEnemy;
            _entityType = eEntityType.eEntityEnemy;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading.
        /// </summary>
        public override void init()
        {
            _iIdTextureGround = Visu.Instance.loadTilset("enemysheet.xml");
            _iIdSpriteGround = Visu.Instance.getSpriteId(_iIdTextureGround, "five_star_shooters_ground");
            _iIdAnimation = Visu.Instance.getAnimationID("Five_Star_Shooters");
            _iIdTextureBullet = _iIdTextureGround;
            _iIdSpriteBullet = Visu.Instance.getSpriteId(_iIdTextureGround, "five_star_shooter_bullet");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Little_Explosion");

            int iSpriteHeight = Visu.Instance.getSpriteHeight(_iIdTextureGround, _iIdSpriteGround);
            int groundWidth = Visu.Instance.getSpriteWidth(_iIdTextureGround, _iIdSpriteGround);
            _shapeGround = Physics.Physics.Instance.createStaticRectangle(groundWidth, iSpriteHeight, Vector2.Zero, _v2Position, 
                0, this);
            _shapeGround._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            int iAnimSprite = Visu.Instance.getAnimation(_iIdAnimation).getIndexSprite(1);
            int iAnimHeight = Visu.Instance.getSpriteHeight(_iIdTextureGround, iAnimSprite);
            int iAnimWidth = Visu.Instance.getSpriteWidth(_iIdTextureGround, iAnimSprite);
            int iBulletWidth = Visu.Instance.getSpriteWidth(_iIdTextureBullet, _iIdSpriteBullet);
            _v2ShootingPosition = _v2Position - Vector2.UnitY * (iAnimHeight - iSpriteHeight);
            _v2BulletSpawnPosition = _v2ShootingPosition + new Vector2(iAnimWidth / 2, 0) - new Vector2(iBulletWidth / 2, 0);

            _shapeShoot = Physics.Physics.Instance.createStaticRectangle(iAnimWidth,
                iAnimHeight, Vector2.Zero, _v2ShootingPosition, 0, this);
            
            _shapeShoot._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            _v2AnimationExplosionOffset = new Vector2(groundWidth / 2, 0);
            base.init();
        }

        /// <summary>
        /// Activator
        /// </summary>
        /// <param name="position"></param>
        public override void activate()
        {
            _innerState = eState.eStateGround;
  
            _shapeGround._bCollisionEnable = true;
            _shapeShoot._bCollisionEnable = false;

            //int iSpriteHeight = Visu.Instance.getSpriteHeight(_iIdTextureGround, _iIdSpriteGround);
            //int iAnimSprite = Visu.Instance.getAnimation(_iIdAnimation).getIndexSprite(1);
            //int iAnimHeight = Visu.Instance.getSpriteHeight(_iIdTextureGround, iAnimSprite);
            //int iAnimWidth = Visu.Instance.getSpriteWidth(_iIdTextureGround, iAnimSprite);
            //int iBulletWidth = Visu.Instance.getSpriteWidth(_iIdTextureBullet, _iIdSpriteBullet);
            //_v2ShootingPosition = _v2Position - Vector2.UnitY * (iAnimHeight - iSpriteHeight);
            //_v2BulletSpawnPosition = _v2ShootingPosition + new Vector2(iAnimWidth/2, 0) - new Vector2(iBulletWidth/2, 0);

            
            _bPreState = true;
            _bActive = true;
            _bAnimationOver = false;
        }

        public override void update()
        {
            CollisionResult res = null;
            switch (_innerState)
            {
                case eState.eStateGround:
                    updateStateGround();
                    res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shapeGround, (int)ePhysicGroup.ePhysicPlayer);
                    break;

                case eState.eStateShoot:
                    updateStateShoot();
                    res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shapeShoot, (int)ePhysicGroup.ePhysicPlayer);
                    break;
            }

             if (res != null)
            {
                World.Instance.PlayerEntity.hurt(_iDamages);
            }
        }

        private void updateStateGround()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iGroundStartTime = TimeClock.Clock.instance.millisecs;
                _shapeGround._bCollisionEnable = true;
                _shapeShoot._bCollisionEnable = false;
            }

            if (TimeClock.Clock.instance.millisecs > _iGroundStartTime + _iGroundTime)
            {
                _innerState = eState.eStateShoot;
                _bPreState = true;
            }
        }

        private void updateStateShoot()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iAnimationCount = 0;
                _bShoot = false;
                _shapeGround._bCollisionEnable = false;
                _shapeShoot._bCollisionEnable = true;
            }

            //check if the animation is over
            if (_bAnimationOver)
                _iAnimationCount++;

            //after two animations, change state
            if (_iAnimationCount == 2)
            {
                _innerState = eState.eStateGround;
                _bPreState = true;
            }

            //shoot after the animation rolled once
            if (!_bShoot)
            {
                if (_iAnimationCount == 1)
                {
                    const int LIFE_TIME = 3000;
                    _bShoot = true;
                    BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, LIFE_TIME, 1, new Vector2(-3, 0), _v2BulletSpawnPosition, eSide.eSideEnemy);
                    BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, LIFE_TIME, 1, new Vector2(-2, -2), _v2BulletSpawnPosition, eSide.eSideEnemy);
                    BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, LIFE_TIME, 1, new Vector2(0, -3), _v2BulletSpawnPosition, eSide.eSideEnemy);
                    BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, LIFE_TIME, 1, new Vector2(2, -2), _v2BulletSpawnPosition, eSide.eSideEnemy);
                    BulletManager.Instance.activate(_iIdTextureBullet, _iIdSpriteBullet, LIFE_TIME, 1, new Vector2(3, 0), _v2BulletSpawnPosition, eSide.eSideEnemy);
                }
            }
    
        }

        public override void render()
        {
            switch (_innerState)
            {
                case eState.eStateGround:
                    Visu.Instance.displaySprite(_iIdTextureGround, _iIdSpriteGround, ScreenPosition);
                    break;

                case eState.eStateShoot:
                    _bAnimationOver = Visu.Instance.displayAnimation(_iIdAnimation, _v2ShootingPosition - World.Instance.CameraPosition, ref _iAnimationCurrentFrame, ref _iAnimationCurrentTime);
                    break;
            }
        }

        public override void die()
        {
            _shapeShoot._bCollisionEnable = false;
            _shapeGround._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position+_v2AnimationExplosionOffset);
        }

        //public override void hurt(int damages)
        //{
        //   // if (!_bActive) return;
        //    die();
        //}
    }
}
